#include <stdio.h>
#include <string.h>

#include "system.h"
#include <sys/alt_irq.h>
#include "altera_avalon_i2c.h"
#include "altera_avalon_pio_regs.h"

//Please Set following Parameter for your system
//#################################################################
#define I2C_NAME            (I2C_0_NAME)    /* From system.h */
#define I2C_DEV_ADDR        (0x7A >> 1)          /* I2C Slave Address(HDMI PHY) */
#define DEBUG_MODE          (0)             /* 0:Disable log print / 1:Enable log print */
//#################################################################

/* Can be removed all print function from here
 * (LightWeight Device Driver Mode has possibility to be stuck when Nios2-terminal is not connected) */
#if DEBUG_MODE
//#define dprintf(...) {printf("%s(%3d): ", __func__, __LINE__); printf(__VA_ARGS__);}
#define dprintf(...) {printf(__VA_ARGS__);}
#else
#define dprintf(...)
#endif

// Function Prototype
int hdmi_init(void);
ALT_AVALON_I2C_STATUS_CODE i2c_write(ALT_AVALON_I2C_DEV_t *i2c_dev, unsigned int target_addr, unsigned int data);

// I2C Sequence Table
unsigned char i2c_reg_list[][2] = {
		/* reg_addr, value */
		{0x98, 0x03},
		{0x01, 0x00},
		{0x02, 0x18},
		{0x03, 0x00},
		{0x14, 0x70},
		{0x15, 0x20},
		{0x16, 0x30},
		{0x18, 0x46},
		{0x40, 0x80},
		{0x41, 0x10},
		{0x49, 0xA8},
		{0x55, 0x10},
		{0x56, 0x08},
		{0x96, 0xF6},
		{0x73, 0x07},
		{0x76, 0x1F},
		{0x98, 0x03},
		{0x99, 0x02},
		{0x9A, 0xE0},
		{0x9C, 0x30},
		{0x9D, 0x61},
		{0xA2, 0xA4},
		{0xA3, 0xA4},
		{0xA5, 0x04},
		{0xAB, 0x40},
		{0xAF, 0x16},
		{0xBA, 0x60},
		{0xD1, 0xFF},
		{0xDE, 0x10},
		{0xE4, 0x60},
		{0xFA, 0x7D},
		{0x98, 0x03}
};

volatile low_stuck_detect;
volatile isr_detect;
static void hdmi_int_isr(void* context, alt_u32 id)
{
	IOWR_ALTERA_AVALON_PIO_DATA(HDMI_INT_PIO_BASE,0x00);
	IOWR_ALTERA_AVALON_PIO_EDGE_CAP(HDMI_INT_PIO_BASE, 0x1);
	IOWR_ALTERA_AVALON_PIO_IRQ_MASK(HDMI_INT_PIO_BASE, 0x1);

	isr_detect = 1;
}

ALT_AVALON_I2C_STATUS_CODE i2c_write(ALT_AVALON_I2C_DEV_t *i2c_dev, unsigned int target_addr, unsigned int data)
{
	volatile ALT_AVALON_I2C_STATUS_CODE status;
	alt_u8 tx_buf[2]={0};
	alt_u32 tx_size = 0;

	tx_buf[tx_size++] = target_addr;
	tx_buf[tx_size++] = data;

	dprintf("target_addr = 0x%02x ",target_addr);
	dprintf("data = 0x%02x ",data);

	status = alt_avalon_i2c_master_tx(i2c_dev,
			tx_buf,
			tx_size,
			0); //use isr?

	dprintf("status = %d \n",status);

	return status;
}

int hdmi_init(void)
{
	ALT_AVALON_I2C_STATUS_CODE status;
	ALT_AVALON_I2C_DEV_t *i2c_dev;
	ALT_AVALON_I2C_MASTER_CONFIG_t i2c_cfg;
	int i;
	int loop_size;

	//Open I2C
	i2c_dev = alt_avalon_i2c_open(I2C_NAME);
	if(i2c_dev == NULL){
		dprintf("I2C Open Error \n");
		return ALT_AVALON_I2C_ERROR;
	}

	//alt_avalon_i2c_master_config_speed_set(i2c_dev, &i2c_cfg, 400000); //set 400 kHz
	alt_avalon_i2c_master_config_speed_set(i2c_dev, &i2c_cfg, 100000); //set 100 kHz
	alt_avalon_i2c_master_config_set(i2c_dev, &i2c_cfg);
	alt_avalon_i2c_master_config_get(i2c_dev, &i2c_cfg);

	//Set I2C Slave Addr
	alt_avalon_i2c_master_target_set(i2c_dev, I2C_DEV_ADDR);


	loop_size = sizeof(i2c_reg_list) / sizeof(i2c_reg_list[0]);
	dprintf("loop_size = %d \n", loop_size);
	for(i=0; i < loop_size; i++){
		status = i2c_write(i2c_dev, i2c_reg_list[i][0], i2c_reg_list[i][1]);
	}

	return status;
}

int main()
{
	ALT_AVALON_I2C_STATUS_CODE status;

	dprintf("First HDMI Setup \n");

	dprintf("Assert HDMI Reset \n");
	IOWR_ALTERA_AVALON_PIO_DATA(HDMI_RESET_PIO_BASE, 0x0);
	dprintf(" HDMI INT = %d \n",IORD_ALTERA_AVALON_PIO_DATA(HDMI_INT_PIO_BASE));

	dprintf("Register HDMI ISR \n");
	isr_detect = 0;
	dprintf(" HDMI INT = %d \n",IORD_ALTERA_AVALON_PIO_DATA(HDMI_INT_PIO_BASE));
	IOWR_ALTERA_AVALON_PIO_DATA(HDMI_INT_PIO_BASE,0x00);
	IOWR_ALTERA_AVALON_PIO_IRQ_MASK(HDMI_INT_PIO_BASE, 0x1);
	IOWR_ALTERA_AVALON_PIO_EDGE_CAP(HDMI_INT_PIO_BASE, 0x1);
	alt_irq_register(HDMI_INT_PIO_IRQ, 0, hdmi_int_isr );

	dprintf("Release HDMI Reset \n");
	IOWR_ALTERA_AVALON_PIO_DATA(HDMI_RESET_PIO_BASE, 0x1);

	usleep(1000);

#if 1
	dprintf("Start HDMI Setup \n");
	do{
		status = hdmi_init();
	} while(status != ALT_AVALON_I2C_SUCCESS);

	dprintf("Done \n");
#endif

	dprintf("Enter Infinite Loop \n");

	while(1){
		if(isr_detect != 0)	{
			dprintf(" Detect ISR \n");
			dprintf("  HDMI INT with IRQ = %d \n",IORD_ALTERA_AVALON_PIO_DATA(HDMI_INT_PIO_BASE));

			do{
				status = hdmi_init();
			} while(status != ALT_AVALON_I2C_SUCCESS);

			isr_detect = 0;

			usleep(10);
		}

		if(IORD_ALTERA_AVALON_PIO_DATA(HDMI_INT_PIO_BASE) != 1)	{
			low_stuck_detect++;

			if(low_stuck_detect >= 100){
				dprintf(" Detect HDMI INT Low Stuck \n");
				dprintf(" HDMI INT = %d \n",IORD_ALTERA_AVALON_PIO_DATA(HDMI_INT_PIO_BASE));

				do{
					status = hdmi_init();
				} while(status != ALT_AVALON_I2C_SUCCESS);

				low_stuck_detect = 0;
			}
		}

		usleep(10);
	}

	return 0;
}


